<?php

namespace App\Http\Controllers;

use App\Models\MissingPerson;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;

class MissingPersonController extends Controller
{
    public function index(Request $request)
    {
        $query = MissingPerson::where('is_verified', true)
                              ->where('status', 'Active');

        // Apply filters
        if ($request->filled('city')) {
            $query->where('last_seen_location', 'like', '%' . $request->city . '%');
        }

        if ($request->filled('gender')) {
            $query->where('gender', $request->gender);
        }

        if ($request->filled('min_age')) {
            $query->where('age', '>=', $request->min_age);
        }

        if ($request->filled('max_age')) {
            $query->where('age', '<=', $request->max_age);
        }

        $missingPersons = $query->latest()->paginate(12);

        return view('missing.index', compact('missingPersons'));
    }

    public function show($id)
    {
        $missingPerson = MissingPerson::where('is_verified', true)->findOrFail($id);
        return view('missing.show', compact('missingPerson'));
    }

    public function create()
    {
        return view('missing.create');
    }

    public function store(Request $request)
    {
        \Log::info('Store request received', $request->all());
        
        $validated = $request->validate([
            'relation' => 'required|string',
            'reporter_contact' => 'required|string',
            'reporter_city' => 'required|string',
            'reporter_state' => 'required|string',
            'name' => 'required|string',
            'age' => 'required|integer|min:1|max:120',
            'gender' => 'required|in:Male,Female,Other',
            'height' => 'nullable|string',
            'complexion' => 'nullable|string',
            'language' => 'nullable|string',
            'date_missing' => 'required|date',
            'last_seen_location' => 'required|string',
            'medical_condition' => 'nullable|string',
            'photo' => 'required|image|max:5120',
            'fir_copy' => 'required|file|mimes:pdf,jpg,jpeg,png|max:5120',
            'identification_marks' => 'nullable|string',
            'clothes' => 'nullable|string',
        ]);

        // Handle file uploads
        if ($request->hasFile('photo')) {
            $validated['photo'] = $request->file('photo')->store('photos', 'public');
        }

        if ($request->hasFile('fir_copy')) {
            $validated['fir_copy'] = $request->file('fir_copy')->store('fir_copies', 'public');
        }

        $validated['user_id'] = Auth::id();
        $validated['status'] = 'Pending Verification';

        MissingPerson::create($validated);

        return redirect()->route('dashboard.index')
                         ->with('success', 'Missing person report submitted successfully. It will be verified soon.');
    }

    public function edit($id)
    {
        $missingPerson = MissingPerson::where('user_id', Auth::id())->findOrFail($id);
        return view('missing.edit', compact('missingPerson'));
    }

    public function update(Request $request, $id)
    {
        $missingPerson = MissingPerson::where('user_id', Auth::id())->findOrFail($id);

        $validated = $request->validate([
            'relation' => 'required|string',
            'reporter_contact' => 'required|string',
            'reporter_city' => 'required|string',
            'reporter_state' => 'required|string',
            'name' => 'required|string',
            'age' => 'required|integer|min:1|max:120',
            'gender' => 'required|in:Male,Female,Other',
            'height' => 'nullable|string',
            'complexion' => 'nullable|string',
            'language' => 'nullable|string',
            'date_missing' => 'required|date',
            'last_seen_location' => 'required|string',
            'medical_condition' => 'nullable|string',
            'photo' => 'nullable|image|max:5120',
            'fir_copy' => 'nullable|file|mimes:pdf,jpg,jpeg,png|max:5120',
            'identification_marks' => 'nullable|string',
            'clothes' => 'nullable|string',
        ]);

        // Handle file uploads
        if ($request->hasFile('photo')) {
            if ($missingPerson->photo) {
                Storage::disk('public')->delete($missingPerson->photo);
            }
            $validated['photo'] = $request->file('photo')->store('photos', 'public');
        }

        if ($request->hasFile('fir_copy')) {
            if ($missingPerson->fir_copy) {
                Storage::disk('public')->delete($missingPerson->fir_copy);
            }
            $validated['fir_copy'] = $request->file('fir_copy')->store('fir_copies', 'public');
        }

        $missingPerson->update($validated);

        return redirect()->route('dashboard.index')
                         ->with('success', 'Missing person report updated successfully.');
    }

    public function markAsFound($id)
    {
        $missingPerson = MissingPerson::where('user_id', Auth::id())->findOrFail($id);
        $missingPerson->update(['status' => 'Resolved']);

        return redirect()->route('dashboard.index')
                         ->with('success', 'Case marked as resolved.');
    }

    public function destroy($id)
    {
        $missingPerson = MissingPerson::where('user_id', Auth::id())->findOrFail($id);
        
        // Delete files
        if ($missingPerson->photo) {
            Storage::disk('public')->delete($missingPerson->photo);
        }
        if ($missingPerson->fir_copy) {
            Storage::disk('public')->delete($missingPerson->fir_copy);
        }

        $missingPerson->delete();

        return redirect()->route('dashboard.index')
                         ->with('success', 'Report deleted successfully.');
    }
}
